# /// script
# requires-python = ">=3.13"
# dependencies = [
#     "kaleido==1.2.0",
#     "marimo",
#     "numpy==2.4.2",
#     "pandas==3.0.0",
#     "plotly==6.5.2",
#     "scipy==1.17.0",
#     "statsmodels==0.14.6",
# ]
# ///

import marimo

__generated_with = "0.19.7"
app = marimo.App(width="medium")


@app.cell(hide_code=True)
def _(mo):
    mo.hstack(
        [
            mo.image(
                src="http://modernmacro.org/resources/Vivaldo/figures/OldTests/RIR.png",
                width=200,
            ),
            mo.image(
                src="http://modernmacro.org/resources/Vivaldo/figures/iscte.png",
                width=300,
            ),
        ],
        align="center",
    )
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    <style>
    .marimo-cell {
        border: none !important;
        box-shadow: none !important;
        margin: 0 !important;
        padding: 0 !important;
    }

    .marimo-cell-content {
        padding: 0.5rem 0 !important;
    }
    .admonition {
    margin-top: -0.99rem !important;
    margin-bottom: -0.99rem !important;
    }
    h1 {
        margin-bottom: -1.2rem !important;
    }
    h2 {
        margin-bottom: -1.2rem !important;
    }
    /* Reduce spacing before bullet lists */
    ul {
        margin-top: -0.3rem !important;
    }
    /* Optionally reduce spacing between list items */
    li {
        margin-bottom: 0.2rem !important;
    }
    /* Increase font size for all text in the document */
    body {
        font-size: 16px;
    }

    /* Increase font size for all headers */
    h1 {
        font-size: 36px !important;
    }

    h2 {
        font-size: 28px !important;
    }

    h3 {
        font-size: 25px !important;
    }
    </style>
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    # Recap: Basic Statistics & Dynamic Processes

    <br>
    **02 February 2026**
    <br>

    **Vivaldo Mendes**
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md("""
    ---
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    ### Packages used in this notebook

    | Package | Role |
    |---|---|
    | `marimo` | Reactive notebook framework |
    | `numpy` | Numerical computing & linear algebra |
    | `pandas` | Data manipulation & CSV I/O |
    | `plotly` | Interactive plotting |
    | `statsmodels` | `acf` / `ccf` (autocorrelation & cross-correlation) |
    """)
    return


@app.cell
def _():
    import marimo as mo
    import numpy as np
    import pandas as pd
    from scipy.optimize import fsolve                    
    import plotly.graph_objects as go                      
    from plotly.subplots import make_subplots
    import plotly.express as px
    import plotly.io as pio 
    pio.templates.default = "plotly"
    import kaleido
    pio.defaults.default_format = "svg"
    from datetime import date, datetime
    from numpy.linalg import eigvals, inv
    from statsmodels.tsa.stattools import acf, ccf as statsmodels_ccf
    import warnings
    warnings.filterwarnings("ignore")
    import os
    os.chdir(os.path.dirname(__file__))
    return acf, eigvals, go, inv, mo, np, pd, px


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    ## 1. Basic statistical concepts
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    This is how most economic data looks like:
    """)
    return


@app.cell(hide_code=True)
def _(go, inflation):
    fig_portugal = go.Figure(
        data=[
            go.Scatter(
                x=inflation["Months"],
                y=inflation["Portugal"],
                mode="markers+lines",
                marker=dict(symbol="circle", size=5, color="darkred"),
                line=dict(width=0.3),
            )
        ]
    )
    fig_portugal.update_layout(
        title_text="Inflation in Portugal",
        title_x=0.5,
        height=450,
        hovermode="x",
        xaxis_range=["1996-08-01", "2024-06-01"],
        margin=dict(l=70, r=60, t=70, b=60)
    )
    fig_portugal
    return


@app.cell(hide_code=True)
def _(inflation, px):
    # Bar plot of inflation deviation from target
    fig8 = px.bar(x=inflation.Months, y=(inflation.EuroArea - 2), color_discrete_sequence=['darkred'] )
    fig8.update_layout(height=450, title_text='EuroArea Inflation Deviation from 2% Target', hovermode = "x", margin=dict(l=70, r=60, t=70, b=60))
    fig8
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    This is how random shocks look like:
    """)
    return


@app.cell(hide_code=True)
def _(go, np):
    np.random.seed(42)
    random_obs = np.random.randn(300)

    fig_random = go.Figure(
        data=[
            go.Scatter(
                y=random_obs,
                mode="markers+lines",
                marker=dict(symbol="circle", size=5, color="darkred"),
                line=dict(width=0.3),
            )
        ]
    )
    fig_random.update_layout(
        height=450,
        title="Two hundred random normally distributed observations",
        title_font_size=18,
        margin=dict(l=70, r=60, t=70, b=60)  
    )
    fig_random
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    ### The sample mean

    For a given real-valued array $X$, the sample *mean* is given by

    $$\overline{x}=\frac{x_{1}+x_{2}+\ldots+x_{n}}{n}=\frac{\sum_{i=1}^{n} x_{i}}{n} \tag{1}$$

    where $n$ is the length of the input, $x_i$ are the observations in $X$, and $\overline{x}$ is the sample mean.
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    #### Sample variance and standard deviation

    The sample *variance* and the sample *standard deviation* are given by, respectively:

    $$s_x^{2}=\operatorname{Var}(x)=\frac{\sum_{i=1}^{n}\left(x_{i}-\bar{x}\right)^{2}}{n-1} \tag{2}$$

    and

    $$s_x=\sqrt{\frac{\sum_{i=1}^{n}\left(x_{i}-\bar{x}\right)^{2}}{n-1}} \tag{3}$$
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    ### The Auto-Correlation function

    To compute the covariance between the vectors $x$ and $y$, we can use the Pearson's **correlation coefficient**. When applied to a sample, the former is commonly represented by $r_{xy}$ and may be referred to as the sample correlation coefficient. We can obtain a formula for $r_{xy}$ by substituting estimates of the covariances and variances based on a sample into the formula above. Given paired data $\left\{\left(x_{1}, y_{1}\right), \ldots,\left(x_{n}, y_{n}\right)\right\}$ consisting of $n$ pairs, $r_{xy}$ is defined as:

    $$r_{xy}=\frac{1}{n-1} \sum_{i=1}^{n}\left(x_{i}-\bar{x}\right)\left(y_{i}-\bar{y}\right) \tag{4}$$

    where:

    - $n$ is sample size

    - $x_{i}, y_{i}$ are the individual sample points indexed with $i$

    - $\bar{x}=\frac{1}{n} \sum_{i=1}^{n} x_{i}$ (the sample mean); and analogously for $\bar{y}$

    The **autocorrelation function** with $k$ lags of vector $y$ is defined as

    $$r_{y}(k)=\frac{\sum_{i=1}^{n-k}\left(y_{i}-\bar{y}\right)\left(y_{i+k}-\bar{y}\right)}{\sum_{i=1}^{n}\left(y_{i}-\bar{y}\right)^{2}} \tag{5}$$
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    ### The Cross-Correlation function

    TTo compute the cross-covariance between two vectors $x_t$ and $y_t$, at the $k$ lag, we do:

    $$cv_{x y}(k)=\frac{1}{n-1} \sum_{t=1}^{n-k}\left(x_{t+k}-\bar{x}\right)\left(y_{t}-\bar{y}\right) \tag{6}$$

    where, according to our notation above, $\{\overline{x},\overline{y}\}$ are the sample means of $x_t$ and $y_t$, respectively, while $k$ is a positive lag.

    The cross-correlation function is a normalized version of the cross-variance by dividing it by the sample variances of $x_{t}$ and $y_{t}$. As the sample variances also have $n-1$ in their denominators, the terms $n-1$ cancel out and the cross-correlation function can be computed using the formula:

    $$cr_{x y}(k)= \frac{\sum_{t=1}^{n-k}\left(x_{t+k}-\bar{x}\right)\left(y_t-\bar{y}\right)} {\sqrt{\sum_{t=1}^n\left(x_t-\bar{x}\right)^2} \times \sqrt{\sum_{t=1}^n\left(y_t-\bar{y}\right)^2}} \tag{7}$$
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    #### Python functions for basic statistical measures

    | Measure | Function | Package |
    |:---|:---|:---|
    | Mean | `np.mean(x)` | NumPy |
    | Standard deviation | `np.std(x, ddof=1)` | NumPy |
    | Auto-correlation | `acf(x, nlags=...)` | statsmodels |
    | Cross-correlation | `crosscor(x, y, lags)` *(custom)* | — |
    | Correlation coefficient | `np.corrcoef(x, y)[0,1]` | NumPy |
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    ### Using random observations

    "*In discrete time, white noise is a discrete signal whose samples are regarded as a sequence of serially
    uncorrelated random observations with zero mean and finite variance; a single realization of white noise
    is a random shock*". — Wikipedia
    """)
    return


@app.cell
def _(np):
    np.random.seed(100)
    noise = np.random.randn(500)
    shrek = np.random.randn(500)
    return noise, shrek


@app.cell(hide_code=True)
def _(go, noise):
    fig_wn = go.Figure(
        data=[
            go.Scatter(
                y=noise,
                mode="markers+lines",
                marker=dict(size=6, color="blue"),
                line=dict(color="blueviolet", width=0.5),
            )
        ]
    )
    fig_wn.update_layout(
        title_text="White noise",
        title_x=0.5,
        hovermode="x",
        height=450,
        margin=dict(l=70, r=60, t=70, b=60)  
    )
    fig_wn
    return


@app.cell
def _(noise, np):
    mean_noise = np.mean(noise)
    mean_noise
    return


@app.cell
def _(noise, np):
    std_noise = np.std(noise, ddof=1)
    std_noise
    return


@app.cell
def _(acf, noise):
    acf_noise = acf(noise, nlags=25, fft=True)
    acf_noise
    return (acf_noise,)


@app.cell(hide_code=True)
def _(acf_noise, go):
    fig_acf_noise = go.Figure(
        data=[
            go.Scatter(
                y=acf_noise,
                mode="markers+lines",
                marker=dict(symbol="circle", size=7, color="blue"),
                line=dict(color="blueviolet", width=0.6),
            )
        ]
    )
    fig_acf_noise.update_layout(
        title_text="Autocorrelation function: white noise",
        title_x=0.5,
        height=450,
        xaxis_title="lag",
        yaxis_title="Autocorrelation",
        margin=dict(l=70, r=60, t=70, b=60)  
    )
    fig_acf_noise
    return


@app.cell(hide_code=True)
def _(go, shrek):
    lag_amount = 1  # try 1, 10, 20, 100
    shrek_lagged = shrek[:-lag_amount] if lag_amount > 0 else shrek
    shrek_orig   = shrek[lag_amount:]  if lag_amount > 0 else shrek

    fig_shrek = go.Figure(
        data=[
            go.Scatter(
                x=shrek_orig,
                y=shrek_lagged,
                mode="markers",
                marker=dict(symbol="circle", size=6, color="blue"),
            )
        ]
    )
    fig_shrek.update_layout(
        width=600, height=600,
        title_text="shrek vs shrek_lagged",
        title_x=0.5,
        xaxis_title="shrek",
        yaxis_title="shrek_lagged",
        margin=dict(l=70, r=60, t=70, b=60)  
    )
    fig_shrek
    return


@app.cell(hide_code=True)
def _(go, noise, shrek):
    fig_ns = go.Figure(
        data=[
            go.Scatter(
                x=noise,
                y=shrek,
                mode="markers",
                marker=dict(symbol="circle", size=6, color="maroon"),
            )
        ]
    )
    fig_ns.update_layout(
        width=600, height=600,
        title_text="noise vs shrek",
        title_x=0.5,
        xaxis_title="noise",
        yaxis_title="shrek",
        margin=dict(l=70, r=60, t=70, b=60)  
    )
    fig_ns
    return


@app.cell
def _(np):
    clags = np.arange(-100, 101)  # -100:100 inclusive
    return (clags,)


@app.cell(hide_code=True)
def _(clags, crosscor, go, noise, shrek):
    ccf_ns = crosscor(noise, shrek, clags) #uses the crosscor function that can be found at the end of this notebook

    fig_cc_ns = go.Figure(
        data=[
            go.Scatter(
                x=clags,
                y=ccf_ns,
                mode="markers+lines",
                marker=dict(symbol="circle", size=6, color="blue"),
                line=dict(width=0.3),
            )
        ]
    )
    fig_cc_ns.update_layout(
        title_text="Cross-correlation between noise and shrek",
        title_x=0.5,
        height=450,
        margin=dict(l=70, r=60, t=70, b=60)  
    )
    fig_cc_ns
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    ### Using data
    """)
    return


@app.cell
def _(pd, preview):
    inflation = pd.read_csv("ECB Data Inflation.csv", parse_dates=["Months"])
    preview(inflation)
    return (inflation,)


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    #### Select a subset of a data frame based on years
    """)
    return


@app.cell
def _(inflation, pd):
    first_date = pd.Timestamp("1997-02-28")
    last_date  = pd.Timestamp("1997-09-30")
    subset_inflation = inflation[
        (inflation["Months"] >= first_date) & (inflation["Months"] <= last_date)
    ]
    subset_inflation
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    #### Another way of doing it
    """)
    return


@app.cell
def _(inflation):
    subset_by_year = inflation[
        (inflation["Months"].dt.year >= 1997) & (inflation["Months"].dt.year <= 1999)
    ]
    subset_by_year
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    #### … still another way of doing it
    """)
    return


@app.cell
def _(inflation, pd):
    subset_range = inflation[
        (inflation["Months"] >= pd.Timestamp("1997-06-30"))
        & (inflation["Months"] <= pd.Timestamp("1998-06-30"))
    ]
    subset_range
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    ### How to create an artificial data set
    """)
    return


@app.cell
def _(np, pd):
    dates = pd.date_range(start="1997-01-31", end="2023-12-31", freq="ME")
    df = pd.DataFrame({"Date": dates, "Value": np.random.rand(len(dates))})
    df
    return (df,)


@app.cell
def _(df, pd):
    start_date = pd.Timestamp("2002-07-31")
    end_date   = pd.Timestamp("2005-06-30")
    subset_df  = df[(df["Date"] >= start_date) & (df["Date"] <= end_date)]
    subset_df
    return


@app.cell
def _(inflation):
    inflation.describe()
    return


@app.cell
def _(inflation):
    inflation.describe(include="all")
    return


@app.cell(hide_code=True)
def _(go, inflation):
    fig_ez_pt = go.Figure()

    fig_ez_pt.add_trace(
        go.Scatter(
            x=inflation["Months"],
            y=inflation["EuroArea"],
            name="EuroZone",
            mode="markers+lines",
            marker=dict(symbol="circle", size=5, color="darkblue"),
            line=dict(width=0.5),
        )
    )
    fig_ez_pt.add_trace(
        go.Scatter(
            x=inflation["Months"],
            y=inflation["Portugal"],
            name="Portugal",
            mode="markers+lines",
            marker=dict(symbol="circle", size=5, color="maroon"),
            line=dict(width=0.5),
        )
    )

    fig_ez_pt.update_layout(
        height=450,
        title ="Inflation in the EuroZone and Portugal",
        title_x=0.5,
        hovermode="x",
        xaxis=dict(
            title="Monthly observations",
            tickformat="%Y",
            hoverformat="%Y-M%m",
        ),
        yaxis_title="Percentage points",
        title_font_size=16,
        margin=dict(l=70, r=60, t=70, b=60)  
    )
    fig_ez_pt
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    To obtain the autocorrelation function, we can use the `acf` function from the `statsmodels` package, as follows:
    """)
    return


@app.cell(hide_code=True)
def _(acf, go, inflation):
    acf_ez = acf(inflation["EuroArea"].values, nlags=25 - 1, fft=True)

    fig_acf_ez = go.Figure(
        data=[
            go.Scatter(
                y=acf_ez,
                mode="markers+lines",
                marker=dict(symbol="circle", size=7, color="blue"),
                line=dict(width=0.3),
            )
        ]
    )
    fig_acf_ez.update_layout(
        title_text="Auto-correlation of inflation in the EZ",
        title_x=0.5,
        hovermode="x",
        height=450,
        margin=dict(l=70, r=60, t=70, b=60)
    )
    fig_acf_ez
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    To plot Inflation against lagged-Inflation, we can use a slider to see what happens with different lags.
    """)
    return


@app.cell
def _(mo):
    lags_slider = mo.ui.slider(-36, 36, value=0, step=1, label="lags", show_value=True)
    lags_slider
    return (lags_slider,)


@app.cell(hide_code=True)
def _(go, inflation, lags_slider):
    EZ = inflation["EuroArea"].values
    lag_val = lags_slider.value

    if lag_val > 0:
        ez_orig   = EZ[lag_val:]
        ez_lagged = EZ[:-lag_val]
    elif lag_val < 0:
        ez_orig   = EZ[:lag_val]
        ez_lagged = EZ[-lag_val:]
    else:
        ez_orig   = EZ
        ez_lagged = EZ

    fig_ez_lag = go.Figure(
        data=[
            go.Scatter(
                x=ez_orig,
                y=ez_lagged,
                mode="markers",
                marker=dict(symbol="circle", size=6, color="#7a5135"),
            )
        ]
    )
    fig_ez_lag.update_layout(
        width=600, height=600,
        title_text="Inflation vs Lagged-Inflation: EZ",
        title_x=0.5,
        xaxis_title="EZ inflation",
        yaxis_title="EZ inflation (lagged)",
        margin=dict(l=70, r=60, t=70, b=60)
    )
    fig_ez_lag
    return


@app.cell(hide_code=True)
def _(go, inflation):
    fig_ez_pt_scatter = go.Figure(
        data=[
            go.Scatter(
                x=inflation["EuroArea"],
                y=inflation["Portugal"],
                mode="markers",
                marker=dict(symbol="circle", size=6, color="blue"),
            )
        ]
    )
    fig_ez_pt_scatter.update_layout(
        width=600, height=600,
        title_text="Inflation in the EuroZone and Portugal",
        title_x=0.5,
        xaxis_title="EuroZone",
        yaxis_title="Portugal",
        yaxis_range=[-1.0, 5.5],
        xaxis_range=[-1.0, 6.5],
        margin=dict(l=70, r=60, t=70, b=60)
    )
    fig_ez_pt_scatter
    return


@app.cell
def _(inflation, np):
    cor_ez_pt = np.corrcoef(inflation["EuroArea"], inflation["Portugal"])[0, 1]
    cor_ez_pt
    return


@app.cell(hide_code=True)
def _(clags, crosscor, go, inflation):
    ccf_ez_pt = crosscor(
        inflation["EuroArea"].values,
        inflation["Portugal"].values,
        clags,
    )

    fig_cc_ez_pt = go.Figure(
        data=[
            go.Scatter(
                x=clags,
                y=ccf_ez_pt,
                mode="markers+lines",
                marker=dict(symbol="circle", size=6, color="blue"),
                line=dict(width=0.3),
            )
        ]
    )
    fig_cc_ez_pt.update_layout(
        title_text="Cross-correlation between inflation in the EuroZone and Portugal",
        title_x=0.5,
        height=450,
        margin=dict(l=70, r=60, t=70, b=60)
    )
    fig_cc_ez_pt
    return


@app.cell
def _(clags, crosscor, inflation, pd):
    ccf_python = crosscor(
        inflation["EuroArea"].values,
        inflation["Portugal"].values,
        clags,
    )
    pd.DataFrame({"ccf_python": ccf_python}).to_csv("ccf_python.csv", index=False)
    type(ccf_python)
    ccf_python
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md("""
    ---
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    ## 2. Stability of dynamic systems
    <br>
    A dynamical process displays the evolution of one variable (or a set of variables) across time, starting from
    a particular initial value (or initial values), resulting from the impact of a set of given forces.

    It has three fundamental ingredients:

    - An initial condition
    - Transitional dynamics
    - A steady state.

    See the following figure for a graphical example.
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.Html(
        '<img src="https://vivaldomendes.org/images/depot/transitional_dynamics.png" width="650" style="max-width:100%;">'
    )
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    Consider the simplest possible dynamic process: an Auto-Regressive process of first order, usually called
    simply by an AR(1):

    $$y_{t+1}= a + \rho \cdot y_t \tag{8}$$

    where $a$ is a constant (or an exogenous force) and $\rho$ is a parameter.
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    👉 a) Pass the model's data $(a,\rho)$ into the notebook.
    """)
    return


@app.cell
def _():
    a = 10
    ρ = 0.5
    return a, ρ


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    👉 **b)** Calculate the steady state of this model.  Remember that the steady state is obtained by imposing
    the condition $y_{t+1} = y_t = \bar{y}$ upon our AR(1).  Therefore,

    $$\bar{y}=a+\rho \bar{y} \Rightarrow \bar{y}=\frac{1}{1-\rho}a=(1-\rho)^{-1} \cdot a \tag{9}$$
    """)
    return


@app.cell
def _(a, ρ):
    ȳ = ((1 - ρ)**(-1)) * a
    ȳ
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    > **Answer (b)**
    >
    > Here
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    👉 **c)** Check the stability of this model.
    """)
    return


@app.cell
def _(ρ):
    # For a scalar AR(1) the single "eigenvalue" is rho itself
    eigenvalue_ar1 = ρ
    eigenvalue_ar1
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    > **Answer (c)**
    >
    > Here
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    👉 **d)** Simulate the model for a period of 30 iterations, with an initial condition of $y_1 = 4$.
    """)
    return


@app.cell
def _(a, np, ρ):
    n = 30                          # number of iterations we want to simulate
    y1 = 4                          # initial condition
    y = np.zeros(n)                 # Allocating space for y
    y[0] = y1                       # initial condition goes in position 0

    for t in range(n - 1):          # begins the "for" loop
        y[t+1] = a + ρ * y[t]       # the loop works through eq. (8): y(t+1) = a + ρy(t)
                                    # ends the for loop
    return (y,)


@app.cell(hide_code=True)
def _(go, y):
    fig_ar1 = go.Figure(
        data=[
            go.Scatter(
                y=y,
                mode="lines+markers",
                marker=dict(symbol="circle", size=7, color="blue"),
                line=dict(width=0.5),
            )
        ]
    )
    fig_ar1.update_layout(
        title_text="The evolution of an AR(1) process",
        title_x=0.5,
        xaxis_title="time",
        yaxis_title="y(t)",
        height=450,
        margin=dict(l=70, r=60, t=70, b=60)
    )
    fig_ar1
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    ### Exercise 1: add noise

    Introduce a series of stochastic (or random) shocks to the AR(1) process above.
    This stochastic variable is defined below as $\varepsilon$ and the process is written as:

    $$y_{t+1}= a + \rho \cdot y_t + \varepsilon_{t+1} \tag{10}$$

    The impact of this random element can be obtained by adding the term `ε[t+1]` to the loop above.
    Now, the loop should be written as:

    ```python
        y[t + 1] = a + rho * y[t] + epsilon[t + 1]
    ```

    To avoid multiple definitions in the notebook, let us name our variable `y_noise` instead of `y`,
    where `y_noise` represents "y with noise".  Everything else will remain the same, apart from telling the
    notebook the kind of noise we want in the model.  In this case, we will use `n` random normally distributed
    observations (`np.random.randn(m)`).
    """)
    return


@app.cell
def _(a, np, ρ):
    m  = 30                                  # number of iterations we want to simulate
    y_noise1 = 4                             # initial condition
    𝜺 = np.random.randn(m)                   # Random shocks
    y_noise = np.zeros(m)                    # Allocating space for y
    y_noise[0] = y_noise1                    # initial condition goes in position 0

    for _t in range(m  - 1):                  # begins the "for" loop
        y_noise[_t+1] = a + ρ * y_noise[_t] + 𝜺[_t+1]  # the loop works through eq. (10): y(t+1) = a + ρy(t) + ε(t+1)
                                            # ends the for loop
    return (y_noise,)


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    Now, we can plot our variable $y$ with noise:
    """)
    return


@app.cell(hide_code=True)
def _(go, y_noise):
    fig_noise = go.Figure(
        data=[
            go.Scatter(
                y=y_noise,
                mode="lines+markers",
                marker=dict(symbol="circle", size=7, color="blue"),
                line=dict(width=0.5),
            )
        ]
    )
    fig_noise.update_layout(
        height=450,
        title_text="The evolution of an AR(1) process with noise",
        title_x=0.5,
        xaxis_title="time",
        yaxis_title="y(t)",
        margin=dict(l=70, r=60, t=70, b=60)
    )
    fig_noise
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    ### Exercise 2: public debt

    The sustainability of public debt is given by this simple linear difference equation:

    $$d_{t+1}=p+\left(\frac{1+r}{1+g}\right) d_{t} \tag{11}$$

    where

    - $d_t$ is the public debt as a percentage of GDP
    - $p$ is the primary deficit as a percentage of GDP
    - $r$ is the real interest rate paid on public debt
    - $g$ is the growth rate of real GDP

    Let us simulate the dynamics associated with this equation using a <kbd>for</kbd> loop.  By changing the sliders below,
    present a scenario that is capable of reducing the level of $d_t$ to 80 % of GDP in no more than fifty years. Notice that the sliders present values in terms of percentage points. Foe example, $r=1.5$ means that $r$ is equal to 1.5 percentage points.
    """)
    return


@app.cell
def _(mo):
    p_slider  = mo.ui.slider(-5.0, 5.0,  value=0.5,  step=0.1,  label="p  … Primary deficit",      show_value=True)
    r_slider  = mo.ui.slider( 0.0, 4.0,  value=1.5,  step=0.05, label="r  … Real interest rate",    show_value=True)
    g_slider  = mo.ui.slider( 0.0, 5.0,  value=2.0,  step=0.05, label="g  … GDP growth rate",       show_value=True)
    d1_slider = mo.ui.slider( 0.0, 2.0,  value=1.2,  step=0.1,  label="d1 … Initial value",         show_value=True)

    mo.hstack([
        mo.vstack([p_slider,  r_slider]),
        mo.vstack([g_slider, d1_slider]),
    ], justify="center", gap=0.1)
    return d1_slider, g_slider, p_slider, r_slider


@app.cell
def _(d1_slider, g_slider, np, p_slider, r_slider):
    N_debt = 500
    p_val  = p_slider.value
    r_val  = r_slider.value
    g_val  = g_slider.value
    d1_val = d1_slider.value

    d_debt = np.zeros(N_debt)
    d_debt[0] = d1_val

    for _t in range(N_debt - 1):
        d_debt[_t + 1] = p_val / 100 + ((1 + r_val / 100) / (1 + g_val / 100)) * d_debt[_t]
    return (d_debt,)


@app.cell(hide_code=True)
def _(d_debt, go):
    fig_debt = go.Figure(
        data=[go.Scatter(y=d_debt, mode="lines", line=dict(width=2.5))]
    )
    fig_debt.update_layout(
        height=450,
        title="The evolution of Public Debt as % of GDP",
        title_x=0.5,
        xaxis_title="Years",
        yaxis_title="Debt / GDP",
        title_font_size=16,
        hovermode="x",
        margin=dict(l=70, r=60, t=70, b=60)
    )
    fig_debt
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    ### Exercise 3: add noise to the primary deficit

    To add noise to the dynamic process that describes the sustainability of public debt, add the term
    $\epsilon_{t+1}$ to eq. (11), where $\epsilon$ is meant to represent a sequence of random factors that
    affect the level of public debt over time:

    $$d_{t+1}=p+\left(\frac{1+r}{1+g}\right) d_{t} + \epsilon_{t+1} \tag{12}$$

    The loop will be written as (to express rates in terms of percentage points):

    ```python
        d[t + 1] = p / 100 + ((1 + r / 100) / (1 + g / 100)) * d[t] + epsilon[t + 1] / 100
    ```
    """)
    return


@app.cell(hide_code=True)
def _(d1_slider, g_slider, np, p_slider, r_slider):
    N_debt_n = 500
    p_val_n  = p_slider.value
    r_val_n  = r_slider.value
    g_val_n  = g_slider.value
    d1_val_n = d1_slider.value

    np.random.seed(42)
    epsilon_debt = np.random.randn(N_debt_n)

    d_debt_noise = np.zeros(N_debt_n)
    d_debt_noise[0] = d1_val_n

    for _t in range(N_debt_n - 1):
        d_debt_noise[_t + 1] = (
            p_val_n / 100
            + ((1 + r_val_n / 100) / (1 + g_val_n / 100)) * d_debt_noise[_t]
            + epsilon_debt[_t + 1] / 100
        )
    return (d_debt_noise,)


@app.cell(hide_code=True)
def _(d_debt_noise, go):
    fig_debt_n = go.Figure(
        data=[go.Scatter(y=d_debt_noise, mode="lines", line=dict(width=1.5))]
    )
    fig_debt_n.update_layout(
        height=450,
        title_text="The evolution of Public Debt as % of GDP with Noise",
        title_x=0.5,
        margin=dict(l=70, r=60, t=70, b=60)
    )
    fig_debt_n
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md("""
    ---
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    ## 3. Higher dimension dynamic models
    <br>

    The stability of a dynamic system depends on the **eigenvalues** of the characteristic matrix of the system.
    In our simple AR(1) model

    $$y_{t+1}=a+\rho \cdot y_{t}$$

    there is only one eigenvalue, which corresponds to parameter $\rho$.  As the example above shows, if
    $|\rho|<1$ the process has a unique and stable equilibrium.  On the other hand, if $|\rho|>1$ the process
    will have an equilibrium, but the former will be unstable.  Finally, if $|\rho|=1$ there will be no
    equilibrium in this process.

    A similar reasoning can be applied to our rather more general model:

    $$Ax_{t+1}= Z + Bx_{t}+ C\varepsilon_{t} \tag{13}$$

    where $A, B , C$ are $n\times n$ matrices, and $x_{t+1}, x_{t},\varepsilon_{t}, Z$ are $n\times 1$ vectors.
    Notice also that in order to simulate the model we have to apply:

    $A^{-1}B=D \ , \ A^{-1}C=E \ , \ A^{-1}Z = H.$

    The point is that now we will **not have** just one eigenvalue.  The number of eigenvalues will be dependent
    on the number of variables in the system (or the number of equations).  For example, if our system has 3
    variables (and consequently, 3 equations) matrix $D$ will have a dimension of 3×3 and there will be three
    eigenvalues ($\lambda_i, i=1,2,3$).  If all $|\lambda_i| <1$, the system will converge to a stable
    equilibrium; if (at least) one of them in modulus is larger than $1$ the system has an unstable equilibrium;
    and no equilibrium if one of them is equal to $1$ in modulus.

    See next figure for a summary of the stability conditions of a dynamic process of **dimension 2** (so we have
    two eigenvalues).  For the case where we have three eigenvalues, the reasoning is similar.
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.Html(
        '<img src="https://ebs.de.iscte.pt/stability.png" width="600" style="max-width:100%;">'
    )
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    ### A Vector Auto-Regressive model of order 3: a VAR(3) model

    Consider the following model

    $$x_{t+1}=D x_{t}+E \varepsilon_{t} \tag{14}$$

    where

    $$x_{t+1} =
      \left[ {\begin{array}{c}
       z_{t+1}\\
       y_{t+1}\\
       v_{t+1}
       \end{array} } \right] \tag{15}$$

    and matrices $D$ and $E$ are given by:


    $$D =  \left[ {\begin{array}{ccc}
       0.97 & 0.10 & -0.05 \\
       -0.3 & 0.98 & 0.05 \\
       0.01 &-0.04 & 0.96
      \end{array} } \right] \qquad , \qquad
    E =
      \left[ {\begin{array}{ccc}
       0 & 0 & 0 \\
       0 & 0 & 0 \\
       0 &0 & 0
      \end{array} } \right] \tag{16}$$


    Finally, please assume that the initial state of our system (or its initial conditions) are: $z_1=1$, $y_1=0$, and $v_1=-1$. For clarity, we can write these conditions using a row vector: $x_1 = [1,0,-1]$.
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    👉 Let us simulate our system for 200 iterations to glimpse its dynamics.  Firstly, we have to pass the
    entries of matrix $D$ into the notebook.  Secondly, we have to write the `for` loop, and finally, we will
    get the plot.
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    Matrix $D$:
    """)
    return


@app.cell
def _(np):
    D = np.array([
        [ 0.97,  0.10, -0.05],
        [-0.30,  0.98,  0.05],
        [ 0.01, -0.04,  0.96],
    ])
    return (D,)


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    The for loop:
    """)
    return


@app.cell
def _(D, np):
    N_var = 200
    x1_var = np.array([1, 0, -1], dtype=float)
    x_var = np.zeros((3, N_var))
    x_var[:, 0] = x1_var

    for _t in range(N_var - 1):
        x_var[:, _t + 1] = D @ x_var[:, _t]
    return (x_var,)


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    The plot:
    """)
    return


@app.cell(hide_code=True)
def _(go, x_var):
    fig_var = go.Figure()
    labels  = ["z(t)", "y(t)", "v(t)"]
    colors  = ["royalblue", "maroon", "red"]

    for i, (label, color) in enumerate(zip(labels, colors)):
        fig_var.add_trace(
            go.Scatter(
                y=x_var[i],
                name=label,
                mode="lines",
                line=dict(color=color),
            )
        )

    fig_var.update_layout(
        hovermode="x",
        title_text="A VAR(3) Model",
        title_x=0.5,
        height=450,
        xaxis_title="time",
        yaxis_title="z(t) , y(t) , v(t)",
        margin=dict(l=70, r=60, t=70, b=60)
    )
    fig_var
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    👉 Checking stability: eigenvalues.
    """)
    return


@app.cell
def _(D, eigvals):
    eigs_D = eigvals(D)
    eigs_D
    return (eigs_D,)


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    Checking whether the condition $\sqrt{\alpha^2+\beta^2}$ is greater than 1, less than 1, or equal to 1:
    """)
    return


@app.cell
def _():
    import math
    Cond_D = math.sqrt(0.982782**2 + 0.181356**2)
    Cond_D
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    An easier way is to use the `abs` function in Python (via NumPy).  It immediately gives the results we are
    looking for:
    """)
    return


@app.cell
def _(eigs_D, np):
    abs_eigs_D = np.abs(eigs_D)
    abs_eigs_D
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    Therefore, as $|\sqrt{0.944435^2-0^2}|<1$, and $|0.999375|<1$, this VAR(3) model is stable.  However, its
    kind of stability is a very special one: it takes a huge amount of time to converge to its steady-state.
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    👉 Finally, simulate the same model with one single change: instead of having $D_{1,1}=0.97$, change it to
    $D_{1,1}=0.99$.  Check the eigenvalues with this new parameter value.
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    ### Exercise 4: a model of dimension 2

    Let's see what happens in the following model with dimension 2, and with a vector including constants $(Z)$:

    $$Y_{t+1}= Z+ B \cdot Y_{t} \tag{17}$$

    $$B=\left[ {\begin{array}{cc}
       1.1 & -0.4  \\
       0.5 & 0.2  \\
       \end{array} } \right] \tag{18}$$

    $$Z = \left[ {\begin{array}{c}
       10  \\
       -25 \\
       \end{array} } \right] \tag{19}$$
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    👉 a) Pass the entries of the matrices $B, Z$ into the notebook.
    """)
    return


@app.cell
def _(np):
    B = np.array([[1.1, -0.4], [0.5, 0.2]])
    return (B,)


@app.cell
def _(np):
    Z = np.array([10, -25], dtype=float)
    return (Z,)


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    👉 **b)** Calculate the steady state of this model.
    """)
    return


@app.cell
def _(B, Z, inv, np):
    I2 = np.eye(len(B))
    Y_bar = inv(I2 - B) @ Z
    Y_bar
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    > **Answer (b)**
    >
    > Here
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    👉 **c)** Check the stability of this model.
    """)
    return


@app.cell
def _(B, eigvals):
    eigs_B = eigvals(B)
    eigs_B
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    > **Answer (c)**
    >
    > Here
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    👉 **d)** Simulate the model for a period of 30 iterations.
    """)
    return


@app.cell
def _(B, Z, np):
    iterations_2d = 30
    Y1_2d = np.array([50, 160], dtype=float)
    Y_2d  = np.zeros((2, iterations_2d))
    Y_2d[:, 0] = Y1_2d

    for _t in range(iterations_2d - 1):
        Y_2d[:, _t + 1] = Z + B @ Y_2d[:, _t]
    return (Y_2d,)


@app.cell(hide_code=True)
def _(Y_2d, go):
    fig_2d = go.Figure()

    fig_2d.add_trace(
        go.Scatter(
            y=Y_2d[0],
            name="y1(t)",
            mode="markers+lines",
            marker=dict(symbol="circle", size=6, color="blue"),
            line=dict(width=0.3),
        )
    )
    fig_2d.add_trace(
        go.Scatter(
            y=Y_2d[1],
            name="y2(t)",
            mode="markers+lines",
            marker=dict(symbol="circle", size=6, color="red"),
            line=dict(width=0.3),
        )
    )

    fig_2d.update_layout(
        hovermode="x",
        title_text="A 2-Dimensional System",
        title_x=0.5,
        height=450,
        xaxis_title="time",
        yaxis_title="y1(t) , y2(t)",
        margin=dict(l=70, r=60, t=70, b=60)
    )
    fig_2d
    return


@app.cell
def _(mo):
    mo.md(r"""
    ## Functions
    """)
    return


@app.cell
def _(pd):
    # Making data frames looking like in Julia and Pluto (first 5 rows + last row)
    def preview(df, n=5):
        if len(df) <= n + 1:
            return df

        ellipsis = pd.DataFrame(
            [["..."] * df.shape[1]],
            columns=df.columns,
            index=["…"],
        )

        return pd.concat([
            df.head(n),
            ellipsis,
            df.tail(1),
        ])
    return (preview,)


@app.cell
def _(np):
    def crosscor(x, y, lags):
        """Cross-correlation at given lags, matching Julia StatsBase.crosscor convention."""
        n = len(x)
        x_demean = x - np.mean(x)
        y_demean = y - np.mean(y)
        denom = np.sqrt(np.sum(x_demean**2) * np.sum(y_demean**2))
        result = np.empty(len(lags))
        for i, k in enumerate(lags):
            if k >= 0:
                result[i] = np.sum(x_demean[k:] * y_demean[:n - k]) / denom if k < n else 0.0
            else:
                result[i] = np.sum(x_demean[:n + k] * y_demean[-k:]) / denom if -k < n else 0.0
        return result
    return (crosscor,)


@app.cell
def _(mo):
    # Global CSS for kbd elements throughout the notebook
    mo.Html("""
        <style>
            kbd {
                background-color: #505050 !important;
                color: #fff !important;
                padding: 3px 6px;
                border-radius: 4px;
                font-family: monospace;
                font-size: 0.9em;
                border: 0px solid #666;
            }
        </style>
    """)
    return


@app.cell
def _():
    #background-color: #546ccc !important; # A color that looks goog on white theme;
    return


if __name__ == "__main__":
    app.run()
